<?php
// ------------------------------------------------------------------------------------------------
// MODULO: WizardAltaUsuario.php
// DESCRIPCIÓN: Wizard vertical para registro de usuarios con configuración dinámica.
// ------------------------------------------------------------------------------------------------

// 1. Seguridad de Sesión
if (!isset($_SESSION['Acceso']) || $_SESSION['Acceso'] !== 'AccesoSiqueSi') {
    // Nota: Si es una petición AJAX esperando JSON, podríamos devolver error JSON, 
    // pero mantenemos el comportamiento visual por defecto.
    echo '<div class="alert alert-danger">Acceso Denegado. Por favor inicie sesión.</div>';
    exit;
}

// ------------------------------------------------------------------------------------------------
// 2. LÓGICA DE PROCESAMIENTO INTERNO (Self-POST Pattern)
// ------------------------------------------------------------------------------------------------
// Detectamos si el JS nos está pidiendo encriptar una contraseña
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action']) && $_POST['action'] === 'hash_password') {
    // Limpiamos buffer por si acaso
    while (ob_get_level()) ob_end_clean();
    header('Content-Type: application/json');
    
    $passRaw = $_POST['password'] ?? '';
    // Aplicamos la función solicitada
    $passHash = password_hash($passRaw, PASSWORD_DEFAULT);
    
    echo json_encode(['status' => 'success', 'hash' => $passHash]);
    exit; // IMPORTANTE: Detenemos la ejecución para no devolver el HTML del wizard
}

// ------------------------------------------------------------------------------------------------
// 3. CONFIGURACIÓN DEL MÓDULO (PHP ARRAYS)
// ------------------------------------------------------------------------------------------------

// A. CAMPOS REQUERIDOS (Validación por paso)
$configRequeridos = [
    1 => ['Nombre', 'EMail', 'Telefono_1'],
    2 => ['User', 'Password', 'IDUg', 'Rol']
];

// B. VALORES PREDEFINIDOS
$configPredefinidos = [
    'IDUg' => $IDU,
    'Estado' => 1,
    'VisibleDatos' => '{"theme":"dark"}',
    'Direccion' => 'Ciudad de México'
];

// C. VALORES BLOQUEADOS
$configBloqueados = [
    'IDUg',
    'Token'
];

// D. VALORES VISIBLES
$configVisibles = [
    'Nombre', 'EMail', 'Telefono_1', 'Telefono_2', 'Direccion', // Paso 1
    'User', 'Password', 'Rol', 'IDUg', 'Estado', 'VisibleDatos', 'Token' // Paso 2
];

// E. CATÁLOGO DE ROLES
$configRoles = [
    'Usuario' => 'Usuario Estándar',
    'Admin' => 'Administrador del Sistema',
    'Supervisor' => 'Supervisor de Área',
    'Ventas' => 'Agente de Ventas',
    'Soporte' => 'Soporte Técnico'
];

// Empaquetado de configuración para JS
$appConfig = [
    'requeridos' => $configRequeridos,
    'predefinidos' => $configPredefinidos,
    'bloqueados' => $configBloqueados,
    'visibles' => $configVisibles,
    'roles' => $configRoles
];
$jsonAppConfig = json_encode($appConfig);
?>

<!-- ESTILOS ESPECÍFICOS (Vertical Wizard) -->
<style>
    #wizard-module .step-container {
        display: flex;
        flex-direction: column;
        position: relative;
        height: 100%;
    }
    #wizard-module .step-container::before {
        content: '';
        position: absolute;
        top: 20px;
        left: 19px;
        width: 2px;
        height: 90%;
        background: #e9ecef;
        z-index: 1;
    }
    #wizard-module .step-item {
        display: flex;
        align-items: center;
        margin-bottom: 40px;
        position: relative;
        z-index: 2;
        cursor: default;
    }
    #wizard-module .step-circle {
        width: 40px;
        height: 40px;
        background: #fff;
        border: 2px solid #e9ecef;
        border-radius: 50%;
        display: flex;
        align-items: center;
        justify-content: center;
        font-weight: bold;
        transition: all 0.3s ease;
        flex-shrink: 0;
    }
    #wizard-module .step-label {
        margin-left: 15px;
        font-weight: 600;
        color: #6c757d;
        transition: color 0.3s;
    }
    #wizard-module .step-item.active .step-circle {
        border-color: #007bff;
        background: #007bff;
        color: #fff;
        box-shadow: 0 0 0 3px rgba(0,123,255,0.2);
    }
    #wizard-module .step-item.active .step-label {
        color: #007bff;
    }
    #wizard-module .step-item.completed .step-circle {
        border-color: #28a745;
        background: #28a745;
        color: #fff;
    }
    #wizard-module .step-item.completed .step-label {
        color: #28a745;
    }
    #wizard-module .step-content {
        display: none;
        animation: fadeInRight 0.4s;
    }
    #wizard-module .step-content.active {
        display: block;
    }
    @keyframes fadeInRight {
        from { opacity: 0; transform: translateX(10px); }
        to { opacity: 1; transform: translateX(0); }
    }
</style>

<div id="wizard-module" class="container-fluid">
    <div class="card shadow-sm">
        <div class="card-header">
            <h5><i class="fas fa-user-plus"></i> Alta de Nuevo Usuario</h5>
        </div>
        <div class="card-body">
            
            <div class="row">
                <!-- COLUMNA IZQUIERDA: INDICADORES VERTICALES -->
                <div class="col-md-3 border-right">
                    <div class="step-container py-4">
                        <div class="step-item active" id="badge-step-1">
                            <div class="step-circle">1</div>
                            <div class="step-label">Datos Personales<small class="d-block text-muted font-weight-normal">Nombre y contacto</small></div>
                        </div>
                        <div class="step-item" id="badge-step-2">
                            <div class="step-circle">2</div>
                            <div class="step-label">Credenciales<small class="d-block text-muted font-weight-normal">Login y Roles</small></div>
                        </div>
                        <div class="step-item" id="badge-step-3">
                            <div class="step-circle">3</div>
                            <div class="step-label">Confirmación<small class="d-block text-muted font-weight-normal">Revisar y Guardar</small></div>
                        </div>
                    </div>
                </div>

                <!-- COLUMNA DERECHA: FORMULARIOS -->
                <div class="col-md-9">
                    <form id="formWizard" class="py-2 px-3">
                        
                        <!-- PASO 1 -->
                        <div class="step-content active" id="step-1">
                            <h5 class="mb-4 text-primary border-bottom pb-2">Paso 1: Información Personal</h5>
                            <div class="row">
                                <div class="col-md-6 form-group field-container" data-field="Nombre">
                                    <label>Nombre Completo *</label>
                                    <input type="text" class="form-control" id="Nombre" placeholder="Ej. Juan Pérez">
                                </div>
                                <div class="col-md-6 form-group field-container" data-field="EMail">
                                    <label>Email *</label>
                                    <input type="email" class="form-control" id="EMail" placeholder="correo@ejemplo.com">
                                </div>
                                <div class="col-md-6 form-group field-container" data-field="Telefono_1">
                                    <label>Teléfono 1 *</label>
                                    <input type="tel" class="form-control" id="Telefono_1" placeholder="55 1234 5678">
                                </div>
                                <div class="col-md-6 form-group field-container" data-field="Telefono_2">
                                    <label>Teléfono 2</label>
                                    <input type="tel" class="form-control" id="Telefono_2" placeholder="Opcional">
                                </div>
                                <div class="col-md-12 form-group field-container" data-field="Direccion">
                                    <label>Dirección</label>
                                    <input type="text" class="form-control" id="Direccion" placeholder="Calle, Número, Colonia">
                                </div>
                                <div class="col-md-12 form-group field-container" data-field="Datos_Fiscales">
                                    <label>Datos Fiscales</label>
                                    <textarea class="form-control" id="Datos_Fiscales" rows="2" placeholder="RFC, Razón Social..."></textarea>
                                </div>
                            </div>
                            <div class="text-right mt-4">
                                <button type="button" class="btn btn-primary px-4" onclick="nextStep(2)">Siguiente <i class="fas fa-arrow-right ml-2"></i></button>
                            </div>
                        </div>

                        <!-- PASO 2 -->
                        <div class="step-content" id="step-2">
                            <h5 class="mb-4 text-primary border-bottom pb-2">Paso 2: Configuración de Cuenta</h5>
                            <div class="row">
                                <div class="col-md-6 form-group field-container" data-field="User">
                                    <label>Usuario (Login) *</label>
                                    <input type="text" class="form-control" id="User" placeholder="Usuario para login">
                                </div>
                                <div class="col-md-6 form-group field-container" data-field="Password">
                                    <label>Contraseña *</label>
                                    <input type="password" class="form-control" id="Password">
                                </div>
                                <div class="col-md-4 form-group field-container" data-field="Rol">
                                    <label>Rol de Sistema</label>
                                    <!-- El select se llenará dinámicamente vía JS -->
                                    <select class="form-control" id="Rol"></select>
                                </div>
                                <div class="col-md-4 form-group field-container" data-field="IDUg">
                                    <label>Grupo (IDUg)</label>
                                    <input type="number" class="form-control" id="IDUg" value="1">
                                </div>
                                <div class="col-md-4 form-group field-container" data-field="Estado">
                                    <label>Estado</label>
                                    <select class="form-control" id="Estado">
                                        <option value="1">Activo</option>
                                        <option value="0">Inactivo</option>
                                    </select>
                                </div>
                                <div class="col-md-12 form-group field-container" data-field="VisibleDatos">
                                    <label>Visibilidad (JSON)</label>
                                    <input type="text" class="form-control" id="VisibleDatos" value="{}">
                                </div>
                                <!-- Token suele ser interno, lo dejamos hidden pero con data-field para control -->
                                <div class="d-none field-container" data-field="Token">
                                    <input type="hidden" id="Token" value="<?php echo bin2hex(random_bytes(4)); ?>">
                                </div>
                            </div>
                            <div class="text-right mt-4">
                                <button type="button" class="btn btn-secondary mr-2" onclick="prevStep(1)">Atrás</button>
                                <button type="button" class="btn btn-primary px-4" onclick="nextStep(3)">Revisar <i class="fas fa-arrow-right ml-2"></i></button>
                            </div>
                        </div>

                        <!-- PASO 3 -->
                        <div class="step-content" id="step-3">
                            <h5 class="mb-4 text-success border-bottom pb-2">Paso 3: Confirmación</h5>
                            <div class="bg-light p-4 rounded mb-3 border">
                                <div class="row">
                                    <div class="col-sm-3 text-muted">Nombre:</div>
                                    <div class="col-sm-9 font-weight-bold mb-2" id="resume-nombre"></div>
                                    <div class="col-sm-3 text-muted">Email:</div>
                                    <div class="col-sm-9 font-weight-bold mb-2" id="resume-email"></div>
                                    <div class="col-sm-3 text-muted">Usuario:</div>
                                    <div class="col-sm-9 font-weight-bold mb-2" id="resume-user"></div>
                                    <div class="col-sm-3 text-muted">Rol:</div>
                                    <div class="col-sm-9 font-weight-bold mb-2" id="resume-rol"></div>
                                </div>
                            </div>
                            <div id="msg-error" class="alert alert-danger d-none"></div>
                            <div id="msg-success" class="alert alert-success d-none"><i class="fas fa-check-circle"></i> Usuario y Credenciales guardados correctamente.</div>

                            <div class="text-right mt-4" id="actions-step-3">
                                <button type="button" class="btn btn-secondary mr-2" onclick="prevStep(2)">Atrás</button>
                                <button type="button" class="btn btn-success px-4" onclick="guardarUsuario()"><i class="fas fa-save ml-2"></i> Confirmar y Guardar</button>
                            </div>
                            <div id="loader-step-3" class="text-center mt-3 d-none">
                                <div class="spinner-border text-primary" role="status"></div>
                                <p class="mt-2 text-muted">Generando Hash de contraseña y guardando...</p>
                            </div>
                        </div>

                    </form>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- LÓGICA JS -->
<script>
    const DB_NAME = 'MS_Usuario'; 
    const API_WRITE = '../librerias/AJAX/Execute.php';
    
    // INYECCIÓN DE CONFIGURACIÓN PHP
    const appConfig = <?php echo $jsonAppConfig; ?>;

    document.addEventListener('DOMContentLoaded', () => {
        initWizard();
    });

    function initWizard() {
        // 1. Cargar Roles Dinámicos
        const rolSelect = document.getElementById('Rol');
        rolSelect.innerHTML = '';
        for (const [val, label] of Object.entries(appConfig.roles)) {
            const option = document.createElement('option');
            option.value = val;
            option.text = label;
            rolSelect.appendChild(option);
        }

        // 2. Valores Predefinidos
        for (const [id, value] of Object.entries(appConfig.predefinidos)) {
            const el = document.getElementById(id);
            if (el) el.value = value;
        }

        // 3. Bloqueos
        appConfig.bloqueados.forEach(id => {
            const el = document.getElementById(id); 
            if (el) {
                el.readOnly = true;
                if (el.tagName === 'SELECT') el.disabled = true;
                el.classList.add('bg-light');
            }
        });

        // 4. Visibilidad
        if (appConfig.visibles && appConfig.visibles.length > 0) {
            const allContainers = document.querySelectorAll('.field-container');
            allContainers.forEach(container => {
                const fieldId = container.getAttribute('data-field');
                if (!appConfig.visibles.includes(fieldId)) {
                    container.classList.add('d-none');
                }
            });
        }
    }

    function nextStep(targetStep) {
        const currentStep = targetStep - 1;
        if (!validateStep(currentStep)) return;
        if (targetStep === 3) populateResume();
        showStep(targetStep);
    }

    function prevStep(step) {
        showStep(step);
    }

    function showStep(stepNumber) {
        document.querySelectorAll('.step-content').forEach(el => el.classList.remove('active'));
        document.getElementById('step-' + stepNumber).classList.add('active');

        for (let i = 1; i <= 3; i++) {
            const badge = document.getElementById('badge-step-' + i);
            badge.classList.remove('active', 'completed');
            if (i < stepNumber) badge.classList.add('completed');
            if (i === stepNumber) badge.classList.add('active');
        }
    }

    function validateStep(stepNumber) {
        const fieldsToCheck = appConfig.requeridos[stepNumber];
        if (!fieldsToCheck) return true;

        let isValid = true;
        let firstError = null;

        fieldsToCheck.forEach(fieldId => {
            const el = document.getElementById(fieldId);
            if (el && el.offsetParent !== null) {
                el.classList.remove('is-invalid');
                if (!el.value.trim()) {
                    el.classList.add('is-invalid');
                    isValid = false;
                    if (!firstError) firstError = el;
                }
            }
        });

        if (firstError) firstError.focus();
        return isValid;
    }

    function populateResume() {
        document.getElementById('resume-nombre').innerText = document.getElementById('Nombre').value;
        document.getElementById('resume-email').innerText = document.getElementById('EMail').value;
        document.getElementById('resume-user').innerText = document.getElementById('User').value;
        
        const rolSelect = document.getElementById('Rol');
        const rolText = rolSelect.options[rolSelect.selectedIndex] ? rolSelect.options[rolSelect.selectedIndex].text : '';
        document.getElementById('resume-rol').innerText = rolText;
    }

	
	
	

	
	async function guardarUsuario() {
    const btn = document.getElementById('actions-step-3');
    const load = document.getElementById('loader-step-3');
    const err = document.getElementById('msg-error');
    btn.classList.add('d-none'); load.classList.remove('d-none'); err.classList.add('d-none');

    const escapeSQL = (s) => String(s).replace(/'/g, "''");

    try {
        // =====================================================================
        // 1) Obtener password y convertirla en Base64 (SIN crypto / digest)
        // =====================================================================
        const hashHex  = document.getElementById('Password').value;
         // <-- lo que se enviará al PHP

        // =====================================================================
        // 2) Enviar al archivo Pass.php como lo pide tu código original
        // =====================================================================
        const resp = await fetch("../librerias/AJAX/Pass.php", {
            method: "POST",
            headers: { "Content-Type": "application/json" },
            body: JSON.stringify({ passwordCliente: hashHex })
        });

        if (!resp.ok) throw new Error("Error en la petición de hashing: " + resp.statusText);
        const d = await resp.json();

        if (!d.hash_guardado) throw new Error("Respuesta inválida del servidor al generar hash.");

        // Este es el hash final ya encriptado por password_hash()
        const PassH = d.hash_guardado;
        console.log("PassH generado:", PassH);

        // =====================================================================
        // 3) Recuperar tus datos normales
        // =====================================================================
        const dataLogin = {
            IDUg: document.getElementById('IDUg').value,
            Rol: document.getElementById('Rol').value,
            User: document.getElementById('User').value,
            Estado: document.getElementById('Estado').value,
            Token: document.getElementById('Token').value,
            VisibleDatos: document.getElementById('VisibleDatos').value
        };

        const IDUg = Number(dataLogin.IDUg) || 0;
        if (IDUg <= 0) throw new Error("IDUg inválido.");

        // =====================================================================
        // 4) SQL del primer INSERT
        // =====================================================================
        const sql1 = `INSERT INTO MS_Usuario.Usuarios_Datos 
            (IDUg, Rol, User, Password, Estado, Token) 
            VALUES (${IDUg}, '${escapeSQL(dataLogin.Rol)}', '${escapeSQL(dataLogin.User)}', '${escapeSQL(PassH)}', ${Number(dataLogin.Estado)}, '${escapeSQL(dataLogin.Token)}')`;

        console.log("SQL1:", sql1);

        const res1 = await executeQuery(sql1);
        if (!res1.last_id) throw new Error("Error DB: No se generó ID Principal.");

        const newIDU = res1.last_id;

        // =====================================================================
        // 5) INSERT 2 Usuarios
        // =====================================================================
        const dataUser = {
            Nombre: document.getElementById('Nombre').value,
            Telefono_1: document.getElementById('Telefono_1').value,
            Telefono_2: document.getElementById('Telefono_2').value,
            EMail: document.getElementById('EMail').value,
            Direccion: document.getElementById('Direccion').value,
            Datos_Fiscales: document.getElementById('Datos_Fiscales').value
        };

        const sql2 = `INSERT INTO MS_Usuario.Usuarios 
            (IDU, Nombre, Telefono_1, Telefono_2, EMail, Direccion) 
            VALUES (${newIDU}, '${escapeSQL(dataUser.Nombre)}', '${escapeSQL(dataUser.Telefono_1)}', '${escapeSQL(dataUser.Telefono_2)}', '${escapeSQL(dataUser.EMail)}', '${escapeSQL(dataUser.Direccion)}')`;

        console.log("SQL2:", sql2);
        await executeQuery(sql2);

        // =====================================================================
        // 6) Mensaje final
        // =====================================================================
        load.classList.add('d-none');
        document.getElementById('msg-success').classList.remove('d-none');

        setTimeout(() => { 
            alert("Usuario creado ID: " + newIDU); 
        }, 1000);

    } catch (e) {
        console.error(e);
        load.classList.add('d-none');
        btn.classList.remove('d-none');
        err.innerText = e.message;
        err.classList.remove('d-none');
    }
}



    async function executeQuery(sql) {
        const formData = new FormData();
        formData.append('sql', sql);
        formData.append('db', DB_NAME);
        const response = await fetch(API_WRITE, { method: 'POST', body: formData });
        if (!response.ok) throw new Error("Error HTTP " + response.status);
        return await response.json();
    }
</script>